/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Copyright (C) 2014 Nanjing University, Nanjing, China
 */
 
package napping.domain.mc;

import java.util.Random;
import napping.core.Action;
import napping.core.State;
import napping.core.Task;

/**
 *
 * @author Qing Da <daq@lamda.nju.edu.cn>
 */
public class MountainCarTask extends Task {

    final public double minPosition = -1.2;
    final public double maxPosition = 0.6;
    final public double minVelocity = -0.07 * 1;
    final public double maxVelocity = 0.07 * 1;
    final public double goalPosition = 0.5;
    final public double accelerationFactor = 0.001 * 1;
    final public double gravityFactor = -0.0025;
    final public double hillPeakFrequency = 3.0;
    //This is the middle of the valley (no slope)
    final public double defaultInitPosition = -0.5d;
    final public double defaultInitVelocity = 0.0d;
    final public double rewardPerStep = -1.0d;
    final public double rewardAtGoal = 0.0d;
    private double transitionNoise = 0.05d;
    private Random random;

    public MountainCarTask(Random rand) {
        this.actionSet = new int[]{0, 1, 2};
        this.random = rand;
    }

    @Override
    public State transition(State s, Action a, Random outRand) {
        Random thisRand = outRand == null ? random : outRand;
        MountainCarState cs = (MountainCarState) s;
        int move = actionSet[a.index];

        double acceleration = accelerationFactor;
        double position = cs.position;
        double velocity = cs.velocity;
//System.out.print(move+" "+position+"\t");
        //Noise should be at most
        double thisNoise = 2.0d * accelerationFactor * transitionNoise * (thisRand.nextDouble() - .5d);

        velocity += (thisNoise + ((move - 1)) * (acceleration)) + getSlope(position) * (gravityFactor);
        if (velocity > maxVelocity) {
            velocity = maxVelocity;
        }
        if (velocity < minVelocity) {
            velocity = minVelocity;
        }
        position += velocity;
        if (position > maxPosition) {
            position = maxPosition;
        }
        if (position < minPosition) {
            position = minPosition;
        }
        if (position == minPosition && velocity < 0) {
            velocity = 0;
        }
//        System.out.println(move+" "+position);
        return new MountainCarState(position, velocity);
    }

    public double getSlope(double queryPosition) {
        return Math.cos(hillPeakFrequency * queryPosition);
    }

    @Override
    public double immediateReward(State s) {
        if (isComplete(s)) {
            return 0;
        } else {
            return -1;
        }
    }

    @Override
    public double[] getSAFeature(State s, Action a) {
        double[] feature = s.extractFeature();
        double[] saFea = new double[feature.length + 1];
        System.arraycopy(feature, 0, saFea, 0, feature.length);
        saFea[saFea.length - 1] = a.index;
        return saFea;
    }

    @Override
    public boolean isComplete(State s) {
        MountainCarState cs = (MountainCarState) s;
        if (cs.position >= goalPosition) {
            return true;
        } else {
            return false;
        }
    }

    @Override
    public State getInitialState() {
        return new MountainCarState(defaultInitPosition + 0.5d * (random.nextDouble() - .5d),
                defaultInitVelocity + .12d * (random.nextDouble() - .5d));
    }
}
